# Autoconf macros for groff
#
# Copyright 1989-2017 Free Software Foundation, Inc.
#           2021-2026 G. Branden Robinson
#
# This file is part of groff, the GNU roff typesetting system.
#
# groff is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# groff is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# Construct a short form of the groff version string.
#
# SHORT_VERSION contains the MAJOR_VERSION, MINOR_VERSION, and a portion
# of the REVISION, separated by dots.  Only the part of REVISION before
# the first '.' is used.  For example, if REVISION is
# '3.real.434-5aafd', then SHORT_VERSION is 'x.yy.3', where x and yy are
# MAJOR_VERSION and MINOR_VERSION, respectively.
AC_DEFUN([GROFF_MAKE_SHORT_VERSION], [
  AC_SUBST([SHORT_VERSION],
	   m4_bregexp(AC_PACKAGE_VERSION,[^\(\w+\.\w+\.\w+\).*$],[\1]))
])

# Verify that the package versioning scheme is useful; GNU troff
# requires a strictly numeric one in the first three components x.y.z.

AC_DEFUN([GROFF_CHECK_VERSION_FORMAT], [
  AC_REQUIRE([GROFF_MAKE_SHORT_VERSION])
  groff_version_format_validity=invalid
  AC_MSG_CHECKING([that groff version string has valid format])
  if expr "$SHORT_VERSION" : \
      '[[0-9]][[0-9]]*\.[[0-9]][[0-9]]*\.[[0-9]][[0-9]]*' \
    > /dev/null
  then
    groff_version_format_validity=valid
  fi
  AC_MSG_RESULT([$SHORT_VERSION $groff_version_format_validity])

  if test "$groff_version_format_validity" = invalid
  then
    AC_MSG_NOTICE([groff's version string must start with three decimal
integers separated by dots.  "$SHORT_VERSION" does not match.
    ])
    AC_MSG_ERROR([Aborting.], 1)
  fi
])

# Locate a print spooler for certain output formats.

AC_DEFUN([GROFF_PRINT], [
  groff_have_spooler=no
  if test -z "$PSPRINT"
  then
    AC_CHECK_PROGS([LPR], [lpr])
    AC_CHECK_PROGS([LP], [lp])
    if test -n "$LPR" && test -n "$LP"
     then
      # HP-UX provides an lpr command that emulates lpr using lp,
      # but it doesn't have lpq; in this case we want to use lp
      # rather than lpr.
      AC_CHECK_PROGS([LPQ], [lpq])
      test -n "$LPQ" || LPR=
    fi
    if test -n "$LPR"
    then
      PSPRINT="$LPR"
    elif test -n "$LP"
    then
      PSPRINT="$LP"
    fi
  fi

  if test -n "$PSPRINT"
  then
    groff_have_spooler="$PSPRINT"
    AC_SUBST([PSPRINT])
    # Figure out DVIPRINT from PSPRINT.
    AC_MSG_CHECKING([option to use when spooling DVI files])
    spooler_option=none
    if test -n "$PSPRINT" && test -z "$DVIPRINT"
    then
      if test "$PSPRINT" = lpr
      then
        spooler_option=-d
        DVIPRINT="$PSPRINT $spooler_option"
      else
        DVIPRINT="$PSPRINT"
      fi
    fi
    AC_SUBST([DVIPRINT])
    AC_MSG_RESULT([$spooler_option])
  fi
])

# If no yacc program is available, we issue an error only if not
# building from a distribution archive, because files generated by a
# yacc are already present there.

AC_DEFUN([GROFF_PROG_YACC], [
  AC_CHECK_PROGS([YACC], ['bison -y' byacc yacc], [missing])
  if ! test -f "$srcdir"/.tarball-version
  then
    if test "$YACC" = missing
    then
      AC_MSG_ERROR([could not find 'bison', 'byacc', or 'yacc'], 1)
    fi
  fi
])

# We need Perl 5.8 or newer.

AC_DEFUN([GROFF_PERL], [
  PERLVERSION=v5.8
  AC_PATH_PROG([PERL], [perl], [no])
  if test "$PERL" = no
  then
    AC_MSG_ERROR([perl executable not found], 1)
  fi
  AX_PROG_PERL_VERSION([$PERLVERSION], true,
    AC_MSG_ERROR([perl version is too old (need $PERLVERSION)], 1))
])

# We need m4 to generate some man pages.

AC_DEFUN([GROFF_PROG_M4], [
  AC_CHECK_PROGS([M4], [m4 gm4], [missing])
  if test "$M4" = missing
  then
    AC_MSG_ERROR([could not find an m4 program], 1)
  fi
])

# We need makeinfo from Texinfo 5.0 or newer, for @codequoteundirected.
# The minor version checking logic is present for future needs.

AC_DEFUN([GROFF_PROG_MAKEINFO], [
  # By default automake will set MAKEINFO to MAKEINFO = ${SHELL} <top
  # src dir>/build-aux/missing makeinfo.  As we need a more precise
  # check of makeinfo version, we don't use it.
  MAKEINFO=
  missing=
  AC_CHECK_PROG([MAKEINFO], [makeinfo], [makeinfo])
  if test -z "$MAKEINFO"
  then
    missing="missing 'makeinfo'"
  else
    AC_MSG_CHECKING([for makeinfo version])
    # We need an additional level of quoting to make sed's regexps
    # work.
    [makeinfo_version=`$MAKEINFO --version 2>&1 \
     | sed -e 's/^.* \([^ ][^ ]*\)$/\1/' -e '1q'`]
    AC_MSG_RESULT([$makeinfo_version])
    # Consider only the first two numbers in version number string.
    makeinfo_version_major=`IFS=.; set x $makeinfo_version; echo ${2}`
    makeinfo_version_minor=`IFS=.; set x $makeinfo_version; echo ${3}`
    makeinfo_version_numeric=`
      expr ${makeinfo_version_major}000 + $makeinfo_version_minor`
    if test $makeinfo_version_numeric -lt 5000
    then
      missing="'makeinfo' is too old; version 5.0 or newer needed"
      MAKEINFO=
    fi
  fi

  groff_have_makeinfo=yes
  test -n "$missing" && groff_have_makeinfo=no
  AC_SUBST([MAKEINFO])
])

# 'makeinfo' and 'texi2dvi' are distributed together, so if the former
# is too old, the latter is too.

AC_DEFUN([GROFF_PROG_TEXI2DVI], [
  AC_REQUIRE([GROFF_PROG_MAKEINFO])
  AC_CHECK_PROG([PROG_TEXI2DVI], [texi2dvi], [texi2dvi], [missing])
  groff_have_texi2dvi=no
  if test "$PROG_TEXI2DVI" != missing && test -n "$MAKEINFO"
  then
    groff_have_texi2dvi=yes
  fi
])

# 'texi2dvi' can be installed without TeX itself, so check for a 'tex'
# executable independently.

AC_DEFUN([GROFF_USE_TEX_CHECK], [
  AC_REQUIRE([GROFF_PROG_TEXI2DVI])
  AC_CHECK_PROG([PROG_TEX], [tex], [found], [missing])
  groff_use_tex=no
  test "$PROG_TEX" = found && groff_use_tex=yes
])

# grohtml needs the following programs to produce images from tbl(1)
# tables and eqn(1) equations.

dnl Any macro that tests $use_grohtml should AC_REQUIRE this.

AC_DEFUN([GROFF_CHECK_GROHTML_PROGRAMS], [
  AC_REQUIRE([GROFF_GHOSTSCRIPT_PATH])

  use_grohtml=no
  missing=
  m4_foreach([groff_prog],
dnl Keep this list of programs in sync with grohtml test scripts.
    [[pnmcrop], [pamcut], [pnmtopng], [pnmtops], [ps2ps]], [
      AC_CHECK_PROG(groff_prog, groff_prog, [found], [missing])
      if test $[]groff_prog = missing
      then
        missing="$missing 'groff_prog'"
      fi
    ])

  test "$GHOSTSCRIPT" = "missing" && missing="'gs' $missing"

  if test -z "$missing"
  then
      use_grohtml=yes
  else
    plural=`set $missing; test $[#] -gt 1 && echo s`
    oxford=`set $missing; test $[#] -gt 2 && echo ,`
    missing=`set $missing
      missing=
      while test $[#] -gt 0
      do
        case $[#] in
          1) missing="$missing$[1]" ;;
          2) missing="$missing$[1]$oxford and " ;;
          *) missing="$missing$[1], " ;;
        esac
        shift
      done
      echo $missing`
    verb=`set $missing
      if test $[#] -gt 1
      then
        echo were
      else
        echo was
      fi`

    grohtml_notice="The program$plural $missing $verb not found in \
\$PATH.

  Consequently, groff's HTML output driver, 'grohtml', will not work
  properly.  It will not be possible to prepare or install
  groff-generated documentation in HTML format.
"
   fi
])


AC_DEFUN([GROFF_GROHTML_PROGRAM_NOTICE], [
  AC_REQUIRE([GROFF_CHECK_GROHTML_PROGRAMS])

  if test "$use_grohtml" = no
  then
    AC_MSG_NOTICE([$grohtml_notice])
  fi
])

AC_DEFUN([GROFF_GROPDF_DEPENDENCIES_CHECK], [
  AC_REQUIRE([GROFF_GHOSTSCRIPT_PATH])
  AC_REQUIRE([GROFF_URW_FONTS_CHECK])

  use_gropdf=no
  gropdf_missing_deps=

  if test "$GHOSTSCRIPT" != missing \
    || test "$groff_have_urw_fonts" = yes
  then
    use_gropdf=yes
  fi

  AC_SUBST([use_gropdf])
])

AC_DEFUN([GROFF_MAKEINFO_PROGRAM_NOTICE], [
  AC_REQUIRE([GROFF_PROG_MAKEINFO])

  if test "$groff_have_makeinfo" = no
  then
    AC_MSG_NOTICE([groff's Texinfo manual will not be generated.

  Because 'makeinfo' is either outdated or missing, the Texinfo manual
  will not be generated in any of its output formats (GNU Info, HTML,
  plain text, TeX DVI, or PDF).

  Get the 'texinfo' package version 5.0 or newer to build the manual.
])
  fi
])


AC_DEFUN([GROFF_GROPDF_PROGRAM_NOTICE], [
  AC_REQUIRE([GROFF_GROPDF_DEPENDENCIES_CHECK])

  if test "$use_gropdf" = no
  then
    AC_MSG_NOTICE(['gropdf' will have reduced function.

  Neither Ghostscript nor URW fonts are available; groff documentation
  thus will not be available in PDF.

  'gropdf' will be able to handle only documents using the standard PDF
  base 14 fonts, plus the 'EURO' font groff supplies, and font embedding
  with its '-e' option (accessed via the 'groff' command with the option
  '-P -e') will not be possible.
])
  fi
])

# Make URW font support configurable.

AC_DEFUN([GROFF_URW_FONTS_SUPPORT], [
  AC_ARG_WITH([urw-fonts],
    [AS_HELP_STRING([--without-urw-fonts],
      [disable support for URW fonts])],
    [urwfontsupport="$withval"])
])

# Make URW font directory location configurable.

AC_DEFUN([GROFF_URW_FONTS_PATH], [
  AC_ARG_WITH([urw-fonts-dir],
    [AS_HELP_STRING([--with-urw-fonts-dir=DIR],
      [search for URW PostScript Type 1 fonts in DIR])],
    [urwfontsdir="$withval"])
])

# Check for availability of URW fonts in the directory specified by the
# user (see GROFF_URW_FONTS_PATH above).  We do NOT search the path of
# directories built into Ghostscript because those fonts lack the
# corresponding AFM files we need to generate groff font description
# files; see afmtodit(1).  Ghostscript's own fonts are treated as the
# "default foundry" and we already provide descriptions of them in
# font/devpdf (along with groff's EURO font).

AC_DEFUN([GROFF_URW_FONTS_CHECK], [
  AC_REQUIRE([GROFF_URW_FONTS_SUPPORT])
  AC_REQUIRE([GROFF_URW_FONTS_PATH])
  AC_REQUIRE([GROFF_GHOSTSCRIPT_PATH])
  groff_have_urw_fonts=no
  urwfontsdir=
  if test "$urwfontsupport" != no
  then
    AC_MSG_CHECKING([for URW fonts in Type 1/PFB format])

dnl Keep this list in sync with font/devpdf/Foundry.in.
    _list_paths="\
      /usr/share/fonts/type1/gsfonts/ \
      /usr/share/fonts/default/Type1/ \
      /usr/share/fonts/default/Type1/adobestd35/ \
      /usr/share/fonts/type1/urw-base35/ \
      /usr/share/fonts/urw-base35 \
      /usr/share/ghostscript/Resource/Font \
      /opt/local/share/fonts/urw-fonts/ \
      /usr/local/share/fonts/ghostscript/"

    if test -n "$urwfontsdir"
    then
      _list_paths="$urwfontsdir"
    fi

dnl Keep this list of font file names in sync with the corresponding
dnl entry in font/devpdf/util/BuildFoundries.pl.
    for k in $_list_paths
    do
      for _font_file in \
        URWGothic-Book \
        URWGothic-Book.t1 \
        URWGothic-Book.pfb \
        URWGothicL-Book.pfb \
        a010013l.pfb
      do
        if test -f $k/$_font_file
        then
          AC_MSG_RESULT([found in $k])
          groff_have_urw_fonts=yes
          urwfontsdir=$k
          break 2
        fi
      done
    done

    if test $groff_have_urw_fonts = no
    then
      AC_MSG_RESULT([none found])
      urwfontsdir=
    fi
  fi

  AC_SUBST([groff_have_urw_fonts])
  AC_SUBST(urwfontsdir)
])

AC_DEFUN([GROFF_URW_FONTS_NOTICE], [
  if test "$urwfontsupport" = yes && test "$groff_have_urw_fonts" = no
  then
    gs_verbiage=
    if test "$GHOSTSCRIPT" != missing
    then
      gs_verbiage=' as well as the search path shown by the
  "'"$GHOSTSCRIPT"' -h" command (if available)'
    fi
    AC_MSG_NOTICE([URW fonts in Type 1/PFB format were not found.

  groff font description files for the URW fonts, used by the 'gropdf'
  output driver, will not be available.  Use and embedding of fonts from
  the 'U' foundry in PDF documents generated by groff will not be
  possible.

  You can obtain the URW base 35 fonts from their GitHub project.  As of
  this writing (2023-02-15), you can find them in the 'fonts' directory
  of the following archives (choose one).

    https://github.com/ArtifexSoftware/urw-base35-fonts/archive/refs/
      tags/20200910.zip
    https://github.com/ArtifexSoftware/urw-base35-fonts/archive/refs/
      tags/20200910.tar.gz

  You may wish to check for a newer release.

    https://github.com/ArtifexSoftware/urw-base35-fonts/releases

  'gropdf' looks for these fonts in several directories specified in
  font/devpdf/Foundry.in$gs_verbiage.

  You will need to "make distclean" and re-run the 'configure' script
  after installing the URW fonts.

  Alternatively, you can pass the option '--with-urw-fonts-dir=DIR'
  to 'configure' to look for them in the directory DIR you specify.
  If found, the 'U' foundry will be available via the '-y' option to
  'gropdf' (accessed via the 'groff' command with the option '-P -y').
    ])
  fi
])


# Check whether the pnm tools accept the -quiet option.

dnl Any macro that tests $pnmtools_quiet should AC_REQUIRE this.

AC_DEFUN([GROFF_PNMTOOLS_CAN_BE_QUIET], [
  AC_REQUIRE([GROFF_CHECK_GROHTML_PROGRAMS])

  AH_TEMPLATE([PNMTOOLS_QUIET], [Command-line option to quiet Netpnm.])

  pnmtools_quiet=

  if test "$use_grohtml" = yes
  then
    AC_MSG_CHECKING([for options to make PNM tools quiet])
    if echo P2 2 2 255 0 1 2 0 | pnmtops -quiet > /dev/null 2>&1
    then
      pnmtools_quiet=-quiet
    fi
    if test -n "$pnmtools_quiet"
    then
      AC_MSG_RESULT([$pnmtools_quiet])
    else
      AC_MSG_RESULT([(none)])
    fi
  fi
  AC_SUBST([pnmtools_quiet])
  AC_DEFINE_UNQUOTED([PNMTOOLS_QUIET], ["$pnmtools_quiet"])
])


# Check whether pnmtops can handle the -nosetpage option.
#
# We require this both for grohtml operation and generation of
# doc/gnu.eps from repository builds.

AC_DEFUN([GROFF_PNMTOPS_NOSETPAGE], [
  AC_REQUIRE([GROFF_PNMTOOLS_CAN_BE_QUIET])

  if test "$pnmtops" = missing
  then
    pnmtops_nosetpage=no
  else
    pnmtops_nosetpage="pnmtops $pnmtools_quiet"
    AC_MSG_CHECKING([whether pnmtops accepts the '-nosetpage' option])
    if echo P2 2 2 255 0 1 2 0 | pnmtops -nosetpage > /dev/null 2>&1
    then
      AC_MSG_RESULT([yes])
      pnmtops_nosetpage="pnmtops $pnmtools_quiet -nosetpage"
    else
      AC_MSG_RESULT([no])
    fi
  fi
  AC_SUBST([pnmtops_nosetpage])
])


# Check location of 'gs'; allow '--with-gs=PROG' option to override.

dnl Any macro that tests $GHOSTSCRIPT should AC_REQUIRE this.

AC_DEFUN([GROFF_GHOSTSCRIPT_PATH],
  [AC_REQUIRE([GROFF_GHOSTSCRIPT_PREFS])
   AC_ARG_WITH([gs],
     [AS_HELP_STRING([--with-gs=PROG],
       [actual [/path/]name of ghostscript executable])],
     [GHOSTSCRIPT=$withval],
     [AC_CHECK_TOOLS(GHOSTSCRIPT, [$ALT_GHOSTSCRIPT_PROGS], [missing])])
   test "$GHOSTSCRIPT" = no && GHOSTSCRIPT=missing])

# Preferences for choice of 'gs' program...
# (allow --with-alt-gs="LIST" to override).

AC_DEFUN([GROFF_GHOSTSCRIPT_PREFS],
  [AC_ARG_WITH([alt-gs],
    [AS_HELP_STRING([--with-alt-gs=LIST],
      [alternative names for ghostscript executable])],
    [ALT_GHOSTSCRIPT_PROGS="$withval"],
    [ALT_GHOSTSCRIPT_PROGS="gs gswin32c gsos2"])
   AC_SUBST([ALT_GHOSTSCRIPT_PROGS])])

AC_DEFUN([GROFF_GHOSTSCRIPT_AVAILABILITY_NOTICE], [
  AC_REQUIRE([GROFF_GHOSTSCRIPT_PATH])

  gs_names=gs
  if test -n "$ALT_GHOSTSCRIPT_PROGS"
  then
    gs_names="$ALT_GHOSTSCRIPT_PROGS"
  fi

  if test "$GHOSTSCRIPT" = missing
  then
    AC_MSG_NOTICE([No Ghostscript program was found in \$PATH.

  It was sought under the name(s) "$gs_names".

  groff documentation will not be available in HTML.

  'grohtml' will have reduced function, being unable to produce
  documents using the 'tbl' preprocessor.
    ])
  fi
])

# Ghostscript version check.  Versions 9.00 <= x < 9.54 suffer from a
# rendering glitch that affects the AT&T troff (and groff) special
# character \(lh; see
#   <https://bugs.ghostscript.com/show_bug.cgi?id=703187>.

AC_DEFUN([GROFF_GHOSTSCRIPT_VERSION_CHECK], [
  AC_REQUIRE([GROFF_GHOSTSCRIPT_PATH])

  if test "$GHOSTSCRIPT" != missing
  then
    AC_MSG_CHECKING([for gs version with good left sidebearing handling])
    ghostscript_notice=
    ghostscript_version_good=
    ghostscript_v_string=`"$GHOSTSCRIPT" -v | sed 1q`
    # Get first word.
    ghostscript_words=`echo "$ghostscript_v_string" | cut -d\  -f1-`

    # If the first word is "GPL", discard it.
    if expr "$ghostscript_words" : "GPL" > /dev/null
    then
      ghostscript_words=`echo "$ghostscript_words" | cut -d\  -f2-`
    fi

    # Only do a version check if the program calls itself Ghostscript.
    if expr "$ghostscript_words" : "Ghostscript" > /dev/null
    then
      ghostscript_version_good=no
      ghostscript_version=`echo "$ghostscript_words" | cut -d\  -f2`
      ghostscript_major=`echo "$ghostscript_version" | cut -d. -f1`
      ghostscript_minor=`echo "$ghostscript_version" | cut -d. -f2`

      if test "$ghostscript_major" -lt 9
      then
        ghostscript_version_good=yes
      elif test "$ghostscript_major" -ge 10
      then
        ghostscript_version_good=yes
      elif test "$ghostscript_minor" -ge 54
      then
        ghostscript_version_good=yes
      fi
    fi

    if test "$ghostscript_version_good" = yes
    then
      ghostscript_version="$ghostscript_version (good)"
    elif test "$ghostscript_version_good" = no
    then
      ghostscript_version="$ghostscript_version (buggy)"
      ghostscript_notice="Buggy version of Ghostscript detected."
    else
      ghostscript_notice="Unable to determine version of Ghostscript."
    fi

    if test -n "$ghostscript_version"
    then
      AC_MSG_RESULT([got $ghostscript_version])
    else
      AC_MSG_RESULT([unable to determine])
    fi
  fi
])

AC_DEFUN([GROFF_GHOSTSCRIPT_VERSION_NOTICE], [
  if test -n "$ghostscript_notice"
  then
    AC_MSG_NOTICE([$ghostscript_notice

  Ghostscript versions 9.00 <= x < 9.54 suffer from a rendering glitch
  that affects the AT&T troff (and groff) special character '\(lh'; see
  <https://bugs.ghostscript.com/show_bug.cgi?id=703187>.  If your
  version of Ghostscript has not been patched to fix this problem, you
  may need to work around it in groff documents you render for the
  PostScript (and, for tbl(1) tables, HTML) output devices.
  ])
  fi
])


# GROFF_CSH_HACK(if hack present, if not present)

AC_DEFUN([GROFF_CSH_HACK],
  [AC_MSG_CHECKING([for csh hash hack])

cat <<EOF >conftest.sh
#!/bin/sh
true || exit 0
export PATH || exit 0
exit 1
EOF

   chmod +x conftest.sh
   if echo ./conftest.sh | (csh >/dev/null 2>&1) >/dev/null 2>&1; then
     AC_MSG_RESULT([yes])
     $1
   else
     AC_MSG_RESULT([no])
     $2
   fi
   rm -f conftest.sh])


AC_DEFUN([GROFF_POSIX],
  [AC_MSG_CHECKING([whether -D_POSIX_SOURCE is necessary])
   AC_LANG_PUSH([C++])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <stdio.h>
extern "C" { void fileno(int); }

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_DEFINE([_POSIX_SOURCE], [1],
	[Define if -D_POSIX_SOURCE is necessary.])],
     [AC_MSG_RESULT([no])])
   AC_LANG_POP([C++])])


# srand() of SunOS 4.1.3 has return type int instead of void

AC_DEFUN([GROFF_SRAND],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([for return type of srand])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <stdlib.h>
extern "C" { void srand(unsigned int); }

       ]])
     ],
     [AC_MSG_RESULT([void])
      AC_DEFINE([RET_TYPE_SRAND_IS_VOID], [1],
	[Define if srand() returns void not int.])],
     [AC_MSG_RESULT([int])])
   AC_LANG_POP([C++])])


# In April 2005, autoconf's AC_TYPE_SIGNAL is still broken.

AC_DEFUN([GROFF_TYPE_SIGNAL],
  [AC_MSG_CHECKING([for return type of signal handlers])
   for groff_declaration in \
     'extern "C" void (*signal (int, void (*)(int)))(int);' \
     'extern "C" void (*signal (int, void (*)(int)) throw ())(int);' \
     'void (*signal ()) ();' 
   do
     AC_COMPILE_IFELSE([
	 AC_LANG_PROGRAM([[

#include <sys/types.h>
#include <signal.h>
#ifdef signal
# undef signal
#endif
$groff_declaration

	 ]],
	 [[

int i;

	 ]])
       ],
       [break],
       [continue])
   done

   if test -n "$groff_declaration"; then
     AC_MSG_RESULT([void])
     AC_DEFINE([RETSIGTYPE], [void],
       [Define as the return type of signal handlers
	('int' or 'void').])
   else
     AC_MSG_RESULT([int])
     AC_DEFINE([RETSIGTYPE], [int],
       [Define as the return type of signal handlers
	('int' or 'void').])
   fi])


AC_DEFUN([GROFF_SYS_NERR],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([for sys_nerr in <errno.h>, <stdio.h>, or <stdlib.h>])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

       ]],
       [[

int k;
k = sys_nerr;

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_DEFINE([HAVE_SYS_NERR], [1],
	[Define if you have sys_nerr in <errno.h>, <stdio.h>, or <stdio.h>.])],
     [AC_MSG_RESULT([no])])
   AC_LANG_POP([C++])])

AC_DEFUN([GROFF_SYS_ERRLIST],
  [AC_MSG_CHECKING([for sys_errlist[] in <errno.h>, <stdio.h>, or <stdlib.h>])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

       ]],
       [[

int k;
k = (int)sys_errlist[0];

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_DEFINE([HAVE_SYS_ERRLIST], [1],
	[Define if you have sys_errlist in <errno.h>, <stdio.h>, or <stdlib.h>.])],
     [AC_MSG_RESULT([no])])])

AC_DEFUN([GROFF_OSFCN_H],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([C++ <osfcn.h>])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <osfcn.h>

       ]],
       [[

read(0, 0, 0);
open(0, 0);

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_DEFINE([HAVE_CC_OSFCN_H], [1],
	[Define if you have a C++ <osfcn.h>.])],
     [AC_MSG_RESULT([no])])
   AC_LANG_POP([C++])])


AC_DEFUN([GROFF_LIMITS_H],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([C++ <limits.h>])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <limits.h>

       ]],
       [[

int x = INT_MIN;
int y = INT_MAX;
int z = UCHAR_MAX;

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_DEFINE([HAVE_CC_LIMITS_H], [1],
	[Define if you have a C++ <limits.h>.])],
     [AC_MSG_RESULT([no])])
   AC_LANG_POP([C++])])

AC_DEFUN([GROFF_TIME_T],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([for declaration of time_t])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <time.h>

       ]],
       [[

time_t t = time(0);
struct tm *p = localtime(&t);

       ]])
     ],
     [AC_MSG_RESULT([yes])],
     [AC_MSG_RESULT([no])
      AC_DEFINE([LONG_FOR_TIME_T], [1],
	[Define if localtime() takes a long * not a time_t *.])])
   AC_LANG_POP([C++])])

AC_DEFUN([GROFF_STRUCT_EXCEPTION],
  [AC_MSG_CHECKING([struct exception])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <math.h>

       ]],
       [[

struct exception e;

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_DEFINE([HAVE_STRUCT_EXCEPTION], [1],
	[Define if <math.h> defines struct exception.])],
     [AC_MSG_RESULT([no])])])

AC_DEFUN([GROFF_ARRAY_DELETE],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([whether ISO C++98 array deletion syntax is supported])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM(, [[

char *p = new char[5];
delete [] p;

       ]])
     ],
     [AC_MSG_RESULT([yes])],
     [AC_MSG_RESULT([no])
      AC_MSG_ERROR([an ISO C++98-conformant compiler is required])])
   AC_LANG_POP([C++])])

AC_DEFUN([GROFF_TRADITIONAL_CPP],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([whether C preprocessor uses pre-ISO C90 syntax])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#define name2(a, b) a/**/b

       ]],
       [[

int name2(foo, bar);

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_MSG_ERROR([an ISO C90-conformant C preprocessor is required])],
     [AC_MSG_RESULT([no])])
   AC_LANG_POP([C++])])

AC_DEFUN([GROFF_WCOREFLAG],
  [AC_MSG_CHECKING([w_coredump])
   AC_RUN_IFELSE([
       AC_LANG_PROGRAM([[

#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>

       ]],
       [[

#ifdef WCOREFLAG
  exit(1);
#else
  int i = 0;
  ((union wait *)&i)->w_coredump = 1;
  exit(i != 0200);
#endif

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_DEFINE(WCOREFLAG, 0200,
	[Define if the 0200 bit of the status returned by wait() indicates
	 whether a core image was produced for a process that was terminated
	 by a signal.])],
     [AC_MSG_RESULT([no])],
     [AC_MSG_RESULT([no])])])


AC_DEFUN([GROFF_BROKEN_SPOOLER_FLAGS],
  [AC_MSG_CHECKING([default value for grops -b option])
   test -n "$BROKEN_SPOOLER_FLAGS" || BROKEN_SPOOLER_FLAGS=0
   AC_MSG_RESULT([$BROKEN_SPOOLER_FLAGS])
   AC_SUBST([BROKEN_SPOOLER_FLAGS])])


AC_DEFUN([GROFF_PAGE], [
  AC_MSG_CHECKING([default paper format])
  whence=
  groff_prefix=$prefix
  test "$prefix" = NONE && groff_prefix=$ac_default_prefix
  papersize=/etc/papersize
  if test -z "$PAGE" && test -r "$papersize"
  then
    sedexpr='s/#.*//;s/[ \t]\{1,\}/ /;s/  *$//;s/^  *//;/^$/d;p'
    PAGE=`sed -n "$sedexpr" "$papersize"`
    test -n "$PAGE" && whence=$papersize
  fi

  if test -z "$PAGE"
  then
    descfile=
    if test -r "$groff_prefix"/share/groff/font/devps/DESC
    then
      descfile=$groff_prefix/share/groff/font/devps/DESC
    elif test -r "$groff_prefix"/lib/groff/font/devps/DESC
    then
      descfile=$groff_prefix/lib/groff/font/devps/DESC
    else
      for f in "$groff_prefix"/share/groff/*/font/devps/DESC
      do
	if test -r "$f"
	then
	  descfile=$f
	  break
	fi
      done
    fi

    if test -n "$descfile"
    then
      if grep -q ['^paperlength[	 ]\{1,\}841890'] "$descfile"
      then
	PAGE=A4
      elif grep -q ['^papersize[	 ]\{1,\}[aA]4'] "$descfile"
      then
	PAGE=A4
      fi
      test -n "$PAGE" && whence=$descfile
    fi
  fi

  if test -z "$PAGE"
  then
    resolvconf=/etc/resolv.conf
    domains=
    if test -r "$resolvconf"
    then
      sedexpr='s/#.*//;s/[ \t]\{1,\}/ /;s/  *$//;s/^  *//;/^$/d;
/^\(domain\|search\)/!d;s/\(domain\|search\) //;p'
      domains=`sed -n "$sedexpr" "$resolvconf"`
    fi
    test -n "$domains" && whence=$resolvconf
    if test -z "$domains"
    then
      domains=`(domainname) 2>/dev/null | tr -d '+'`
      test -n "$domains" && whence="'domainname' command"
      if test -z "$domains" || test "$domains" = '(none)'
      then
        domains=`(hostname) 2>/dev/null | grep '\.'`
        test -n "$domains" && whence="'hostname' command"
      fi
    fi
    # resolv.conf's "search" directive might return multiple domains.
    # If any top-level domain is two letters and it's not 'us' or 'ca',
    # assume the system uses A4 paper.
    for d in $domains
    do
      case "$d" in
      [*.[Uu][Ss]|*.[Cc][Aa])]
        ;;
      [*.[A-Za-z][A-Za-z])]
        PAGE=A4 ;;
      esac
    done
  fi

  message=
  test -n "$PAGE" && test -n "$whence" \
    && message=" (inferred from $whence)"

  if test -z "$PAGE"
  then
    PAGE=letter
  fi

  AC_MSG_RESULT([$PAGE$message])
  AC_SUBST([PAGE])
])


AC_DEFUN([GROFF_CXX_CHECK],
  [AC_REQUIRE([AC_PROG_CXX])
   AC_LANG_PUSH([C++])
   if test "$cross_compiling" = no; then
     AC_MSG_CHECKING([that C++ compiler can compile simple program])
   fi
   AC_RUN_IFELSE([
       AC_LANG_SOURCE([[

int main() {
  return 0;
}

       ]])
     ],
     [AC_MSG_RESULT([yes])],
     [AC_MSG_RESULT([no])
      AC_MSG_ERROR([a working C++ compiler is required])],
     [:])

   if test "$cross_compiling" = no; then
     AC_MSG_CHECKING([that C++ static constructors and destructors are called])
   fi
   AC_RUN_IFELSE([
       AC_LANG_SOURCE([[

extern "C" {
  void _exit(int);
}

int i;
struct A {
  char dummy;
  A() { i = 1; }
  ~A() { if (i == 1) _exit(0); }
};

A a;

int main()
{
  return 1;
}

       ]])
     ],
     [AC_MSG_RESULT([yes])],
     [AC_MSG_RESULT([no])
      AC_MSG_ERROR([a working C++ compiler is required])],
     [:])

   AC_MSG_CHECKING([that header files support C++])
   AC_LINK_IFELSE([
       AC_LANG_PROGRAM([[

#include <stdio.h>

       ]],
       [[

fopen(0, 0);

       ]])
     ],
     [AC_MSG_RESULT([yes])],
     [AC_MSG_RESULT([no])
      AC_MSG_ERROR([header files do not support C++
		   (if you are using a version of gcc/g++ earlier than 2.5,
		   you should install libg++)])])
   AC_LANG_POP([C++])])


AC_DEFUN([GROFF_TMAC],
  [AC_MSG_CHECKING([file name prefix of system macro packages])
   sys_tmac_prefix=
   sys_tmac_file_prefix=
   for d in /usr/share/lib/tmac /usr/lib/tmac
   do
     for t in "" tmac.
     do
       for m in an s m
       do
         f=$d/$t$m
         if test -z "$sys_tmac_prefix" \
            && test -f $f \
            && grep '^\.if' $f >/dev/null 2>&1
         then
           sys_tmac_prefix=$d/$t
           sys_tmac_file_prefix=$t
         fi
       done
     done
   done
   sys_tmac_prefix_result='(none)'
   test -z "$sys_tmac_prefix" \
     || sys_tmac_prefix_result="$sys_tmac_prefix"
   AC_MSG_RESULT([$sys_tmac_prefix_result])
   AC_SUBST([sys_tmac_prefix])

   AC_MSG_CHECKING([for system macro packages to make available])
   tmac_wrap=
   space=
   if test "$sys_tmac_file_prefix" = tmac.
   then
     for f in $sys_tmac_prefix*
     do
       suff=`echo $f | sed -e "s;$sys_tmac_prefix;;"`
       case "$suff" in
       e)
         ;;
       *)
         if ! grep "Copyright.*Free Software Foundation" $f >/dev/null
         then
           tmac_wrap="$tmac_wrap$space$suff"
           space=' '
         fi ;;
       esac
     done
   elif test -n "$sys_tmac_prefix"
   then
     files=`echo $sys_tmac_prefix*`
     grep "\\.so" $files >conftest.sol
     for f in $files
     do
       case "$f" in
       ${sys_tmac_prefix}e)
         ;;
       *.me)
         ;;
       */ms.*)
         ;;
       *)
         b=`basename $f`
         if grep "\\.so.*/$b\$" conftest.sol >/dev/null \
            || grep -l "Copyright.*Free Software Foundation" $f \
               >/dev/null
         then
           :
         else
           suff=`echo $f | sed -e "s;$sys_tmac_prefix;;"`
           case "$suff" in
           tmac.*)
             ;;
           *)
             tmac_wrap="$tmac_wrap$space$suff"
             space=' ' ;;
           esac
         fi
       esac
     done
     rm -f conftest.sol
   fi
   tmac_wrap_result='(none)'
   test -z "$tmac_wrap" || tmac_wrap_result="$tmac_wrap"
   AC_MSG_RESULT([$tmac_wrap_result])
   AC_SUBST([tmac_wrap])
])


# Searching if a non-GNU Troff is installed.  The built-in register
# \n[.g] is always 1 in GNU Troff.
AC_DEFUN([GROFF_G], [
  g=
  AC_MSG_CHECKING([for existing troff installation])
  if test "`(echo .tm '|n(.g' | tr '|' '\\\\' | troff -z -i 2>&1) \
    2>/dev/null`" = 0
  then
    AC_MSG_RESULT([yes])
    g=g
  else
    AC_MSG_RESULT([no])
  fi
  AC_SUBST([g])
])


# Controllable groff compatibility wrappers for vendor-provided macro sets
#
#   Background
#
#     groff's configure script checks for a non-GNU system troff implementation
#     in GROFF_G above, and uses this to rename programs that would conflict
#     with the system *roff implementation. On such systems, groff's version of
#     troff is installed as gtroff, groff's version of nroff is installed as
#     gnroff, and so forth. See the ENVIRONMENT section of groff(1)'s manual
#     page for the entry on GROFF_COMMAND_PREFIX, and the SEE ALSO section for
#     details. The implementation of this can be found in "We use Automake's
#     Uniform Naming Scheme" in Makefile.am, and its use of NAMEPREFIX.
#
#     groff's configure script also checks for vendor-provided macro sets in
#     GROFF_TMAC above, which sets 'sys_tmac_prefix' to the location of any
#     vendor-provided macro sets. The script then determines if any of these
#     vendor-provided macro sets are non-GNU implementations. A space-separated
#     list of macro sets requiring compatibility wrappers is stored in
#     'tmac_wrap'.
#
#     If 'tmac_wrap' is a non-empty list, the make infrastructure will build
#     groff compatibility wrappers in the 'tmac' directory for each mentioned
#     macro set. These wrappers enable compatibility mode (.cp 1) and include
#     (.so <filepath>) the vendor implementation of that macro set. The 'an'
#     wrapper gets special treatment, by including the 'andoc.tmac' macro set,
#     before loading the vendor implementation.
#
#     In groff version 1.22.3 and prior, if 'tmac_wrap' is a non-empty list at
#     installation time (make install), the groff compatibility wrappers are
#     installed in '<prefix>/lib/groff/site-tmac' directory (technically
#     'systemtmacdir' in Makefile.am) as the base macro set name (example:
#     an.tmac), and the groff implementation of these macro sets are installed
#     with a 'g' prefix in the '<prefix>/share/groff/<version>/tmac' directory
#     (example: gan.tmac).
#
#     If your system has these compatibility wrappers installed, then the
#     command 'groff -man ...' (as an example) uses your vendor's 'an' macro
#     set in compatibility mode, while the command 'groff -mgan ...' uses the
#     groff implementation of the 'an' macro set.
#
#     With groff being the principal (but not the only) full-featured *roff
#     implementation in use today, and most FLOSS Unix-like Operating Systems
#     using groff as their *roff implementation, the automatic implementation
#     of compatibility wrappers on systems having their own macro sets needs to
#     be revisited. Most FLOSS software is now developed on systems having no
#     other *roff implementation, so manual pages (for example) are developed
#     and proofread on systems using groff's implementation of the 'an' macro
#     set.
#
#     On systems with their own macro sets (especially 'an'), documents can be
#     rendered incorrectly from the document author's point of view, when
#     these compatibility mode wrappers are used. groff's own manual pages are
#     the perfect case in point, with most not showing the principal usage line
#     of a command, due to the use of .SY/.OP/.YS macros to display such lines.
#
#   Updating groff's use of compatibility mode wrappers
#
#     Given the above, on systems with their own (non-GNU) macro sets, groff's
#     AUTOMATIC installation of these generated vendor compatibility wrappers
#     as the principal implementation of a macro set for groff(1) users may not
#     be desirable, especially if the users are principally using groff(1) to
#     format FLOSS manual pages.
#
#     The '--with-compatibility-wrappers=<value>' option to 'configure' gives
#     these systems control over the default macro set implementation in use
#     when using the groff family of tools.
#
#     The supported values of the '--with-compatibility-wrappers=<value>'
#     option are:
#       'check'  [default] Check if system has vendor-provided non-GNU
#                implementation of any *roff macro set. Based on that use the
#                'yes' or 'no' value for '--with-compatibility-wrappers'. This
#                effectively emulates prior groff releases' implementation.
#       'yes'    Install the vendor compatibility wrappers as groff's default
#                implementation of these macro sets. This is compatible with
#                groff implementations up to and including version 1.22.3.
#                Direct use of the 'yes' option is discouraged, as it will
#                abort the configure process if no vendor-provided non-GNU
#                macro set implementations exists.
#       'no'     Do not install any vendor compatibility wrappers. Use the
#                groff implementation of all macro sets as the default.
#       'manual' Install the vendor compatibility wrappers as '<macro>-os.tmac'
#                in the same directory as 'yes' option ('systemtmacdir').
#                groff users can access the vendor-provided macro set via the
#                '-m<macro>-os' option on the command line of a tool from the
#                groff family. The naming of the option value reflects that now
#                a manual step must be taken to use the vendor-provided
#                implementation of the macro set. This option can safely be
#                used on systems without vendor-provided non-GNU macro set
#                implementations to allow for cross-platform build
#                instructions.
#

AC_DEFUN([GROFF_WITH_COMPATIBILITY_WRAPPERS],
  [AC_ARG_WITH([compatibility-wrappers],
    [AS_HELP_STRING([--with-compatibility-wrappers[={check|manual|no|yes}]],
      [choose whether and how groff compatibility wrappers for \
       vendor-provided non-GNU macro packages are installed. \
       'check' searches for such packages, and behaves as if 'yes' or \
       'no' were given, accordingly. \
       'yes' generates compatibility wrappers for the vendor-provided \
       packages to allow their use with groff and related tools. \
       The compatibility wrappers are installed using the
       vendor-provided names, while their groff implementations are \
       installed with a 'g' prefix. \
       Use of the latter thus requires options of the form
       '-mg<package>' (example: -mgan). \
       'no' installs only the groff implementations.
       'manual' generates compatibility wrappers for the \
       vendor-provided packages as '<macro>-os'. \
       Use of the latter thus requires options of the form
       '-m<macro>-os' option (example: -man-os).])],
    [compatibility_wrappers="$withval"],
    [compatibility_wrappers="check"])

    if    test "$compatibility_wrappers" != check \
       && test "$compatibility_wrappers" != yes   \
       && test "$compatibility_wrappers" != no    \
       && test "$compatibility_wrappers" != manual
    then
         AC_MSG_WARN([Invalid '--with-compatibility-wrappers' argument: '$compatibility_wrappers' - assuming 'check' requested.])
         compatibility_wrappers="check"
    fi

    if test -z "$tmac_wrap"
    then
        # No Operating System Macro Sets Present
        if   test "$compatibility_wrappers" = check
        then
            compatibility_wrappers="no"
        elif test "$compatibility_wrappers" = yes
        then
            AC_MSG_ERROR([No non-GNU macro sets found - cannot create and install compatibility wrappers])
        elif test "$compatibility_wrappers" = no
        then
            : # No action required
        elif test "$compatibility_wrappers" = manual
        then
            # 'manual' allows quiet conversion to 'no' to support
            # cross-platform build instructions
            compatibility_wrappers="no"
        fi
    else
        # One or more Operating System Macro Sets Present
        if   test "$compatibility_wrappers" = check
        then
            compatibility_wrappers="yes"
        fi
    fi

    # Now compatibility_wrappers can only be yes, no or manual
    AC_SUBST([compatibility_wrappers])
  ])

# We need the path to install-sh to be absolute.

AC_DEFUN([GROFF_INSTALL_SH],
  [AC_REQUIRE([AC_CONFIG_AUX_DIR_DEFAULT])
   ac_dir=`cd $ac_aux_dir; pwd`
   ac_install_sh="$ac_dir/install-sh -c"])


# Test whether install-info is available.

AC_DEFUN([GROFF_INSTALL_INFO],
  [AC_CHECK_PROGS([INSTALL_INFO], [install-info], [:])])


# At least one Unix system, Apple Macintosh Rhapsody 5.5,
# does not have -lm ...

AC_DEFUN([GROFF_LIBM],
  [AC_CHECK_LIB([m], [sin], [LIBM=-lm])
   AC_SUBST([LIBM])])


# ... while the MinGW implementation of GCC for Microsoft Win32
# does not seem to have -lc.

AC_DEFUN([GROFF_LIBC],
  [AC_CHECK_LIB([c], [main], [LIBC=-lc])
   AC_SUBST([LIBC])])


# Check whether Windows scripts like 'afmtodit.cmd' should be installed.

AC_DEFUN([GROFF_CMD_FILES],
  [AC_MSG_CHECKING([whether to install .cmd wrapper scripts for Windows])
   case "$host_os" in
   *mingw*)
     make_winscripts=winscripts
     AC_MSG_RESULT([yes]) ;;
   *)
     make_winscripts=
     AC_MSG_RESULT([no]) ;;
   esac
   AC_SUBST([make_winscripts])])


# Check whether we need a declaration for a function.
#
# Stolen from GNU bfd.

AC_DEFUN([GROFF_NEED_DECLARATION],
  [AC_MSG_CHECKING([whether $1 must be declared])
   AC_LANG_PUSH([C++])
   AC_CACHE_VAL([groff_cv_decl_needed_$1],
     [AC_COMPILE_IFELSE([
	  AC_LANG_PROGRAM([[

#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_MATH_H
#include <math.h>
#endif

	  ]],
	  [[

#ifndef $1
  char *p = (char *) $1;
#endif

	  ]])
      ],
      [groff_cv_decl_needed_$1=no],
      [groff_cv_decl_needed_$1=yes])])
   AC_MSG_RESULT([$groff_cv_decl_needed_$1])
   if test $groff_cv_decl_needed_$1 = yes; then
     AC_DEFINE([NEED_DECLARATION_]translit($1, [a-z], [A-Z]), [1],
       [Define if your C++ doesn't declare ]$1[().])
   fi
   AC_LANG_POP([C++])])


# If mkstemp() isn't available, use our own mkstemp.cpp file.

AC_DEFUN([GROFF_MKSTEMP],
  [AC_MSG_CHECKING([for mkstemp])
   AC_LANG_PUSH([C++])
   AC_LIBSOURCE([mkstemp.cpp])
   AC_LINK_IFELSE([
       AC_LANG_PROGRAM([[

#include <stdlib.h>
#include <unistd.h>
int (*f) (char *);

       ]],
       [[

f = mkstemp;

       ]])
     ],
     [AC_MSG_RESULT([yes])
      AC_DEFINE([HAVE_MKSTEMP], [1], [Define if you have mkstemp().])],
     [AC_MSG_RESULT([no])
      _AC_LIBOBJ([mkstemp])])
   AC_LANG_POP([C++])])


# Test whether <inttypes.h> exists, doesn't clash with <sys/types.h>,
# and declares uintmax_t.  Taken from the fileutils package.

AC_DEFUN([GROFF_INTTYPES_H],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([C++ <inttypes.h>])
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[

#include <sys/types.h>
#include <inttypes.h>

       ]],
       [[

uintmax_t i = (uintmax_t)-1;

       ]])
     ],
     [groff_cv_header_inttypes_h=yes
      AC_DEFINE([HAVE_CC_INTTYPES_H], [1],
	[Define if you have a C++ <inttypes.h>.])],
     [groff_cv_header_inttypes_h=no])
   AC_MSG_RESULT([$groff_cv_header_inttypes_h])
   AC_LANG_POP([C++])])


# Test for working 'unsigned long long'.  Taken from the fileutils package.

AC_DEFUN([GROFF_UNSIGNED_LONG_LONG],
  [AC_LANG_PUSH([C++])
   AC_MSG_CHECKING([for unsigned long long])
   AC_LINK_IFELSE([
       AC_LANG_PROGRAM([[

unsigned long long ull = 1;
int i = 63;
unsigned long long ullmax = (unsigned long long)-1;

       ]],
       [[

return ull << i | ull >> i | ullmax / ull | ullmax % ull;

       ]])
     ],
     [groff_cv_type_unsigned_long_long=yes],
     [groff_cv_type_unsigned_long_long=no])
   AC_MSG_RESULT([$groff_cv_type_unsigned_long_long])
   AC_LANG_POP([C++])])


# Define uintmax_t to 'unsigned long' or 'unsigned long long'
# if <inttypes.h> does not exist.  Taken from the fileutils package.

AC_DEFUN([GROFF_UINTMAX_T],
  [AC_REQUIRE([GROFF_INTTYPES_H])
   if test $groff_cv_header_inttypes_h = no; then
     AC_REQUIRE([GROFF_UNSIGNED_LONG_LONG])
     test $groff_cv_type_unsigned_long_long = yes \
	  && ac_type='unsigned long long' \
	  || ac_type='unsigned long'
     AC_DEFINE_UNQUOTED([uintmax_t], [$ac_type],
       [Define uintmax_t to 'unsigned long' or 'unsigned long long' if
	<inttypes.h> does not exist.])
   fi])


# Identify PATH_SEPARATOR character to use in GROFF_FONT_PATH and
# GROFF_TMAC_PATH which is appropriate for the target system (POSIX=':',
# MS-DOS/Win32=';').
#
# The logic to resolve this test is already encapsulated in
# '$srcdir/src/include/nonposix.h'.

AC_DEFUN([GROFF_TARGET_PATH_SEPARATOR],
  [AC_MSG_CHECKING([separator character to use in groff search paths])
   cp "$srcdir"/src/include/nonposix.h conftest.h
   AC_COMPILE_IFELSE([
       AC_LANG_PROGRAM([[
	
#include <ctype.h>
#include "conftest.h"

       ]],
       [[

#if PATH_SEP_CHAR == ';'
make an error "Path separator is ';'"
#endif

       ]])
     ],
     [GROFF_PATH_SEPARATOR=":"],
     [GROFF_PATH_SEPARATOR=";"])
   AC_MSG_RESULT([$GROFF_PATH_SEPARATOR])
   AC_SUBST(GROFF_PATH_SEPARATOR)])


# Check for X11.

AC_DEFUN([GROFF_X11], [
  AC_REQUIRE([AC_CANONICAL_HOST])
  AC_REQUIRE([AC_PATH_XTRA])
  groff_no_x=$no_x
  if test -z "$groff_no_x"
  then
    OLDCFLAGS=$CFLAGS
    OLDLDFLAGS=$LDFLAGS
    OLDLIBS=$LIBS
    CFLAGS="$CFLAGS $X_CFLAGS"
    LDFLAGS="$LDFLAGS $X_LIBS"
    LIBS="$LIBS $X_PRE_LIBS -lX11 $X_EXTRA_LIBS"
    X_AW_DEPS=

    LIBS="$LIBS -lXaw"
    AC_MSG_CHECKING([for Xaw library and header files])
    AC_LINK_IFELSE([
      AC_LANG_PROGRAM([[

#include <X11/Intrinsic.h>
#include <X11/Xaw/Simple.h>

        ]],
        [])
      ],
      [
        AC_MSG_RESULT([yes])
        # Modern versions of Xaw depend on the Xpm library and the SHAPE
        # extension.  AIX's linker doesn't figure this out on its own.
        case $host_os in
          aix*) X_AW_DEPS="-lXpm -lXext" ;;
        esac
      ],
      [
        AC_MSG_RESULT([no])
        groff_no_x=yes
      ])

    AC_SUBST([X_AW_DEPS])

    LIBS="$LIBS -lXmu"
    AC_MSG_CHECKING([for Xmu library and header files])
    AC_LINK_IFELSE([
      AC_LANG_PROGRAM([[

#include <X11/Intrinsic.h>
#include <X11/Xmu/Converters.h>

        ]],
        [])
      ],
      [AC_MSG_RESULT([yes])],
      [AC_MSG_RESULT([no])
      groff_no_x=yes])

    CFLAGS=$OLDCFLAGS
    LDFLAGS=$OLDLDFLAGS
    LIBS=$OLDLIBS
  fi

  if ! test "$groff_no_x" = yes
  then
    XDEVDIRS="font/devX75 font/devX75-12 font/devX100 font/devX100-12"
    XPROGDIRS="src/devices/xditview src/utils/xtotroff"
    XLIBDIRS="src/libs/libxutil"
  fi

  AC_SUBST([XDEVDIRS])
  AC_SUBST([XPROGDIRS])
  AC_SUBST([XLIBDIRS])
])


# Interpret the '--with-appdefdir' command-line option.

dnl Don't quote AS_HELP_STRING!
dnl
dnl TODO: Include the computed default in the RHS of the help string.
AC_DEFUN([GROFF_APPDEFDIR_OPTION], [
  AC_ARG_WITH([appdefdir],
    AS_HELP_STRING([--with-appdefdir=DIR],
      [place X11 application defaults files in DIR]))
])


# Get a default value for the application defaults directory.
#
# We ignore the 'XAPPLRES' and 'XUSERFILESEARCHPATH' environment
# variables.
#
# By default if --with-appdefdir is not used, we will install the
# gxditview application defaults in $prefix/lib/X11/app-defaults.
#
# Note that if --with-appdefdir was passed to 'configure', no prefix is
# added to 'appdefdir'.

AC_DEFUN([GROFF_APPDEFDIR_DEFAULT],
  [if test -z "$groff_no_x"; then
     if test -z "$with_appdefdir"; then
       if test "$prefix" = NONE; then
         appdefdir=$ac_default_prefix/lib/X11/app-defaults
       else
         appdefdir=$prefix/lib/X11/app-defaults
       fi
     else
       appdefdir=$with_appdefdir
     fi
   fi
   AC_SUBST([appdefdir])])

# Emit warning if --with-appdefdir hasn't been used.

AC_DEFUN([GROFF_APPDEFDIR_NOTICE],
  [if test -z "$groff_no_x"; then
     if test -z "$with_appdefdir"; then
       AC_MSG_NOTICE([Default X11 application defaults directory \
assumed.

  The application defaults files for gxditview (GXditview and
  GXditview-color) will be installed in the following directory.

    $appdefdir

  To install elsewhere, say, '/etc/X11/app-defaults', add
  '--with-appdefdir=/etc/X11/app-defaults' to the configure script
  command-line options and rerun it (the 'prefix' value has no effect on
  a --with-appdefdir option).

  If the gxditview app-defaults are installed in a directory that is not
  one of the default X11 directories for this purpose (common defaults
  are /usr/lib/X11/app-defaults, /usr/share/X11/app-defaults, and
  /etc/X11/app-defaults), you will have to set the environment variable
  XFILESEARCHPATH to this path.  More details can be found in the X(7)
  manual page, or in the document "X Toolkit Intrinsics - C Language
  Interface manual".
       ])
     fi
   fi])


AC_DEFUN([GROFF_LIBPROGRAMDIR_DEFAULT],
  libprogramdir=$libdir/groff
  AC_SUBST([libprogramdir]))


AC_DEFUN([GROFF_GLILYPONDDIR_DEFAULT],
  glilypond_dir=$libprogramdir/glilypond
  AC_SUBST([glilypond_dir]))


AC_DEFUN([GROFF_GPINYINDIR_DEFAULT],
  gpinyin_dir=$libprogramdir/gpinyin
  AC_SUBST([gpinyin_dir]))


AC_DEFUN([GROFF_REFERDIR_DEFAULT],
  referdir=$libprogramdir/refer
  AC_SUBST([referdir]))

# Generation of doc/gnu.eps requires xpmtoppm.

AC_DEFUN([GROFF_PROG_XPMTOPPM],
  [AC_CHECK_PROG([XPMTOPPM], [xpmtoppm], [found], [missing])])

# Check for make built-in variable RM.

AC_DEFUN([GROFF_MAKE_DEFINES_RM], [
  AC_MSG_CHECKING(whether 'make' defines 'RM')
  make=make
  if test -n "$MAKE"
  then
    make=$MAKE
  fi
  cat <<EOF > test_make_rm.mk
all:
	@if test -n "\$(RM)"; \
	then \
	   echo yes; \
	else \
	   echo no; \
	fi
EOF
  groff_make_defines_rm=`"$make" -sf test_make_rm.mk`
  AC_MSG_RESULT([$groff_make_defines_rm])
  rm -f test_make_rm.mk
])

# Check if diff has option -D, for gdiffmk. If not, check if gdiff is
# available on the system and make the same test. If either diff or
# gdiff is working, it is set to DIFF_PROG.  If -D option is not
# available, DIFF_PROG is left to diff (gdiffmk will report a
# problem).

AC_DEFUN([GROFF_DIFF_D],
  [AC_MSG_CHECKING(for a diff program that supports option -D)
  groff_has_diff_d_option=no
  DIFF_PROG=diff
  diff -Dx /dev/null /dev/null >/dev/null 2>&1 && groff_has_diff_d_option=yes
  if test "$groff_has_diff_d_option" = no; then
    AC_CHECK_PROGS([GDIFF], [gdiff])
    if test -n "$GDIFF"; then
      "$GDIFF" -Dx /dev/null /dev/null >/dev/null 2>&1 && groff_has_diff_d_option=yes
      if test "$groff_has_diff_d_option" = yes; then
        DIFF_PROG="$GDIFF"
      fi
    fi
  fi
  AC_MSG_RESULT([$DIFF_PROG])
  AC_SUBST([DIFF_PROG])])


# Check if 'test' supports the option `-ef`.  POSIX Issue 8 (2024)
# mandates it.  It could be a shell builtin or a separate executable; we
# don't care as long as it works.
#
# We want to truly test /bin/sh here, because that's what our script
# programs identify as the interpreter unless overridden by
# POSIX_SHELL_PROG; determination of that depends on _this_ test.
#
# Using plain 'sh' leads to false positives because Autoconf is good at
# locating a capable shell.  But we want to test the rattletrap jalopy
# that a system might have installed as /bin/sh (hello, Solaris 10).

AC_DEFUN([GROFF_PROG_TEST_SUPPORTS_EF_OPTION],
  [AC_MSG_CHECKING(whether 'test' supports '-ef' option)
  test_ef_works=no
  /bin/sh -c 'test /dev/null -ef /dev/null > /dev/null 2>&1' \
   && test_ef_works=yes
  AC_MSG_RESULT([$test_ef_works])
])


# gdiffmk needs a working 'test' `-ef` option.  If one is not available
# in the default /bin/sh or /bin/test, use Bash to get it.
#
# This test could be made more general by testing other POSIX Issue 8
# (or earlier) shell features, if we happen to rely upon them.  Solaris
# 10 /bin/sh is notoriously incapable.

AC_DEFUN([GROFF_PROG_SH], [
  AC_REQUIRE([GROFF_PROG_TEST_SUPPORTS_EF_OPTION])
  POSIX_SHELL_PROG=no
  if test "$test_ef_works" = no
  then
    # Try to find a shell that is likely to be more modern than the
    # system's /bin/sh; otherwise programs must complain at runtime if
    # the environment is non-conforming to POSIX.
    AC_PATH_PROGS([POSIX_SHELL_PROG], [bash ksh ksh93 ksh88 mksh dash \
                                       ash yash], [no])
  fi
  if test "$POSIX_SHELL_PROG" = no
  then
    POSIX_SHELL_PROG=/bin/sh
  fi
  AC_SUBST([POSIX_SHELL_PROG])
])


# Search for uchardet library used by preconv.

AC_DEFUN([GROFF_UCHARDET], [
  AC_ARG_WITH([uchardet],
    AS_HELP_STRING([--with-uchardet={auto|no|yes}],
      [build 'preconv' against uchardet library to automatically \
detect input file encoding]))
    AS_IF([test "$with_uchardet" != no],
      [PKG_CHECK_MODULES([UCHARDET],
        [uchardet >= 0.0.1],
        [AC_DEFINE([HAVE_UCHARDET], [1],
          [uchardet library availability])
        groff_have_uchardet=yes], [
          if test "$with_uchardet" = yes
          then
            AC_MSG_FAILURE([could not find uchardet library])
          fi
          groff_have_uchardet=no])],
      [groff_have_uchardet=no])
])

AC_DEFUN([GROFF_UCHARDET_NOTICE], [
  if test "$groff_have_uchardet" = no && test "$with_uchardet" != no
  then
    AC_MSG_NOTICE([The uchardet library was not found.

  The 'preconv' preprocessor program will be unable to attempt automatic
  inference of an input file's character encoding.  See the preconv(1)
  man page.
  ])
  fi
])


AC_DEFUN([GROFF_USE_GROFF_ALLOCATOR], [
  AC_ARG_ENABLE([groff-allocator],
    [AS_HELP_STRING([--enable-groff-allocator], [enable libgroff's \
allocator for C++ new/delete])],
    [test "$enableval" = yes && groff_use_own_allocator=yes],
    [groff_use_own_allocator=no])
])
